<?php

namespace WPCWV;


if (!defined('ABSPATH')) {
    die('Let\'s not do this');
}

class cwv_cache
{


    public static function init()
    {
        // clear cache when post is saved
        add_action('save_post', 'WPCWV\cwv_cache::save_post_callback');
        add_action('woocommerce_update_product',  'WPCWV\cwv_cache::woocommerce_update_product_callback');

        // clear cache buttons
        add_action('wp_ajax_critical_clear_cache', 'WPCWV\cwv_cache::ClearHTMLCache');

        // preload cache, page by page
        add_action('cwv_preload_single_page', 'WPCWV\cwv_cache::preloadSinglePage', 10, 1);
    }


    public static function preloadSinglePage($url)
    {

        // dont bother with query params
        if (stristr($url, '?') && cwv_options::get('wp_cwv_page_cache_urlparam') == 0) {
            return true;
        }

        // open url in wp
        wp_remote_get($url);

        // maybe we need to cache this page as a repeat view?
        if (cwv_options::get('wp_cwv_css') == 1) {
            if (cwv_options::get('wp_cwv_critcal_css_only_new') == 1) {

                $cookies = [];
                $cookies[] = new \WP_Http_Cookie(array(
                    'name'  => 'cwv_rv',
                    'value' => 1,
                ));

                $args = [
                    'cookies' => $cookies,
                ];
                wp_remote_get($url, $args);
            }
        }
    }


    /**
     * Clear page cache when woocommerce product is saved!
     */
    public static function woocommerce_update_product_callback($product_id)
    {
        self::clearSinglePageById($product_id);
    }


    /**
     * Clear page cache when post is saved!
     */
    public static function save_post_callback($post_id)
    {
        self::clearSinglePageById($post_id);

        // clear the bloglist page cause that might have changed
        $page_for_posts = get_option('page_for_posts');
        self::clearSinglePageById($page_for_posts);


        // clear all special pages listed by user, maybe we can set some sane defaults??
        self::clearSpecialPages();
    }

    /**
     * Clear page cache by ID!
     * @param $post_id
     * @return void
     */
    private static function clearSinglePageById($post_id)
    {
        $page = get_permalink($post_id);
        self::clearSinglePageByURl($page);
    }

    private static function clearSinglePageByURl($url)
    {
        $url != '';
        $path = str_replace(WP_CWV_SITE_URL, '', trim($url));
        self::clearSinglePageByPath($path);
    }

    private static function clearSinglePageByPath($path)
    {
        $aCacheInfo = cwvh::getCachePathInfo($path);

        // 4 possible combinations, delete them all
        @unlink($aCacheInfo['abspath'] . $aCacheInfo['filename']);
        @unlink($aCacheInfo['abspath'] . 'ccss_' . $aCacheInfo['filename']);

        // preload the cache
        wp_schedule_single_event(time() + 10, 'cwv_preload_single_page', [WP_CWV_SITE_URL . $page]);
    }


    /**
     * Clear special pages once a page has been edited!
     * @return void
     */
    private static function clearSpecialPages()
    {

        // Get the pages to be cleared from the cache
        $pageslines = cwv_options::get('wp_cwv_pages_to_clear');
        // Split the list of pages into an array
        $aPages = explode("\n", $pageslines);
        if (is_array($aPages) && count($aPages) > 0) {
            foreach ($aPages as $page) {
                // Remove any extra spaces from the page url
                $page = trim($page);
                self::clearSinglePageByURl($page);
            }
        }
    }



    public static function ClearCloudFlareCache()
    {
        $zoneId = cwv_options::get('cwv_cf_zone');
        $apiToken = cwv_options::get('cwv_cf_token');

        $url = "https://api.cloudflare.com/client/v4/zones/{$zoneId}/purge_cache";

        $headers = array(
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $apiToken
        );
        $body = array(
            'purge_everything' => true
        );

        $response = wp_remote_post($url, array(
            'headers' => $headers,
            'body' => json_encode($body)
        ));
    }

    /* remove all css and html cache */
    public static function ClearHTMLCache()
    {
        $files = glob(WP_CONTENT_DIR . '/cache/wp_cwv/css/*');

        foreach ($files as $file) {
            if (is_file($file)) {
                unlink($file);
            }
        }

        $files = glob(WP_CONTENT_DIR . '/cache/wp_cwv/html/*');

        foreach ($files as $file) {
            if (is_file($file)) {
                unlink($file);
            }
        }

        self::ClearCloudFlareCache();

        die('done');
    }

    /*
    * Delete all cache files that are older than the set time
    * Called by cron (WPCWV\cwv_admin_install::wp_cwv_hourly_cron())
    */
    public static function deleteOutDatedCache()
    {

        $hours = cwv_options::get('wp_cwv_cache_ttl_hours');
        if ($hours > 0) {
            $files = glob(WP_CONTENT_DIR . '/cache/wp_cwv/html/*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    if (time() - filemtime($file) > $hours * 3600) {
                        unlink($file);
                    }
                }
            }
        }
    }



   



    // quick and dirty check if we need to bypass the cache!
    public static function tryCache($timing = 'early')
    {

        $_SERVER['CWV_REQUEST_URI'] = $_SERVER['REQUEST_URI'];

        // strip tracking parameters
        $urlParts = parse_url($_SERVER['CWV_REQUEST_URI']);
        $query = [];
        if (isset($urlParts['query'])) {

            $ignoreQs = explode(',', cwv_options::get('wp_cwv_cache_ignore_qs'));
            array_push($ignoreQs, 'fbclid', '_hsmi', '_hsenc', 'gclid');

            parse_str($urlParts['query'], $query);
            foreach ($query as $key => $value) {
                if (strpos($key, 'utm_') === 0) {
                    unset($query[$key]);
                } else if (in_array($key, $ignoreQs)) {
                    unset($query[$key]);
                }
            }
            $newQuery = http_build_query($query);
            $_SERVER['CWV_REQUEST_URI'] = $urlParts['path'] . ($newQuery ? '?' . $newQuery : '');
        }

        // check if the request is a preload request, in that case set an expires header 30 secs into the future!
        if (function_exists('getallheaders')) {
            $aHeaders = getallheaders();
        } else {
            $aHeaders = [];

            foreach ($_SERVER as $name => $value) {
                if (substr($name, 0, 5) == 'HTTP_') {
                    $aHeaders[str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))))] = $value;
                }
            }
        }

        if (isset($aHeaders['purpose']) && $aHeaders['purpose'] == 'prefetch') {
            header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 30) . ' GMT');
        }

        if (isset($aHeaders['X-Moz']) && $aHeaders['X-Moz'] == 'prefetch') {
            header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 30) . ' GMT');
        }

        /* late caching, return for now and try later */
        if ($timing == 'early' && cwv_options::get('wp_cwv_late_cache')) {
            return true;
        }

        if (WP_CWV_DEBUG) {
            return true;
        }

        if (parse_url(WP_CWV_SITE_URL, PHP_URL_HOST) != $_SERVER['HTTP_HOST']) {
            return true;
        }

        //items in cart
        if (isset($_COOKIE['woocommerce_items_in_cart'])) {
            return true;
        }

        //logged in
        foreach ((array)$_COOKIE as $cookie_key => $cookie_value) {
            if (preg_match("/wordpress_logged_in/i", $cookie_key)) {
                return true;
            }
        }

        // no cache system
        if (cwv_options::get('wp_cwv_page_cache') == false) {
            return true;
        }


        // Only GET withour vars
        if ($_SERVER['REQUEST_METHOD'] != 'GET') {
            return true;
        } else  if ((is_array($query) && sizeof($query) > 0) || (isset($_GET) && sizeof($_GET) > 0)) {
            if (cwv_options::get('wp_cwv_page_cache_urlparam') == 0) {
                return true;
            }
        }

        // WP-CLI
        if (defined('WP_CLI') && WP_CLI) {
            return true;
        }

        // Other CLI
        // if (isset($_SERVER['argv']) && is_array($_SERVER['argv']) && sizeof($_SERVER['argv']) > 0) {
        //     return true;
        // }

        $filename = md5(trim($_SERVER['HTTP_HOST']) . trim($_SERVER['CWV_REQUEST_URI'], '/')) . '.html';


        if (file_exists(WP_CONTENT_DIR . '/cache/wp_cwv/html/' . $filename . '-link')) {
            $link = file_get_contents(WP_CONTENT_DIR . '/cache/wp_cwv/html/' . $filename . '-link');
            header('Link: ' . $link);
        }

        if (file_exists(WP_CONTENT_DIR . '/cache/wp_cwv/html/' . $filename)) {
            header('X-CWV-Cache: HIT');


            readfile(WP_CONTENT_DIR . '/cache/wp_cwv/html/' . $filename);
            echo "\n<!-- cached and optimized (php cache) by wp-core-web-vitals plugin @ www.corewebvitals.io -->";
            exit();
        }
    }
}
